var HID = require('node-hid');

var vend = 0x24CD,          // these are the expected USB device values
    prod = 0x0008,
    pgmBytes = 0x38;

var reqType = {
    QUERY_DEVICE:   0x02,
    UNLOCK_CONFIG: 0x03,      // NONPUBLIC
    ERASE_DEVICE: 0x04,
    PROGRAM_DEVICE: 0x05,
    PROGRAM_COMPLETE: 0x06,
    GET_DATA: 0x07,
    RESET_DEVICE: 0x08,
    SIGN_FLASH: 0x09,         // NONPUBLIC
    QUERY_EXTENDED_INFO: 0x0C // NONPUBLIC
};

// NONPUBLIC
var devFam = {          // NONPUBLIC NOTE: struct in MCU code mislabels this field as "word size" or similar
    PIC18: 0x01,        // NONPUBLIC
    PIC24: 0x02,        // NONPUBLIC [controller should match this!]
    PIC32: 0x03,        // NONPUBLIC
    PIC16: 0x04         // NONPUBLIC
}, to_devFam = [null, 'PIC18', 'PIC24', 'PIC32', 'PIC16'];      // NONPUBLIC
// NONPUBLIC
var memType = {               // NONPUBLIC
    INITVEC_MEMORY: 0x00,     // NONPUBLIC
    PROGRAM_MEMORY: 0x01,     // NONPUBLIC
    EEPROM_MEMORY: 0x02,      // NONPUBLIC
    CONFIG_MEMORY: 0x03,      // NONPUBLIC
    USERID_MEMORY: 0x04       // NONPUBLIC
}, to_memType = ['INITVEC_MEMORY','PROGRAM_MEMORY','EEPROM_MEMORY','CONFIG_MEMORY','USERID_MEMORY'];    // NONPUBLIC
// NONPUBLIC
// NONPUBLIC

function Modchip(device) {
    this._device = device;
    this.info = {stale:true};
}

Modchip.prototype._pgmBytes = pgmBytes;         // HACK: really should have a better name (move to info?)

Modchip.prototype.refreshInfo = function (cb) {
    this.info.stale = true;
    var info = {};
    this.exchange('QUERY_DEVICE', function (e, d) {
        if (e) return cb(e);
        else if (d[1] !== this._pgmBytes) return cb(new Error("Unexpected programming packet size!"));
        else if (d[2] !== devFam.PIC24) return cb(new Error("Unexpected device type!"));    // NONPUBLIC
        else if (d[2] !== 0x02) return cb(new Error("Unexpected query result."));
        
        info.ranges = [];
        for (var i = 0; i < 6; ++i) {
            var off = 3 + i*(1+4+4),
                type = (d[off] < 0xFF) ? 'ENUMERATED' : void 0,
                type = to_memType[d[off]],      // NONPUBLIC
                addr = d.readUInt32LE(off+1),
                size = d.readUInt32LE(off+5);
            if (typeof type === 'undefined') break;
            if (Modchip._debug) console.log("Modchip has",size,"bytes of",type,"memory at",'0x'+addr.toString(16));
            info.ranges.push({idx:i,type:type,addr:addr,size:size});
        }
        
        var firmwareVersion = d.readUInt32LE(d.length-11);      // (raw value is 16-bit, but transfered in 32-bit field)
        
        d = d.slice(-7);
        var deviceId = d.readUInt32LE(0),
            hardwareId = d.readUInt16LE(0+4),
            bootloaderVersion = d.readUInt8(0+6);
        if (Modchip._debug) console.log("Device ID is",deviceId.toString(16),"and is hardware",hardwareId.toString(16));
        if (Modchip._debug) console.log("Device has bootloader",bootloaderVersion,"and firmware",firmwareVersion);
        
        info.deviceId = deviceId;
        info.hardwareId = hardwareId;
        info.bootloaderVersion = bootloaderVersion;
        info.firmwareVersion = firmwareVersion;
        
        this.info = info;
        if (cb) cb.call(this, null, info);
    }.bind(this));

};

Modchip.prototype.exchange = function (req, data, cb) {
    if (typeof data === 'function') {
        cb = data;
        data = null;
    }
    try {
        this.write(req,data);
    } catch (e) {
        process.nextTick(cb.bind(null,e));
    }
    this.read(function (e,d) {
        if (e) cb(e);
        else if (d[0] !== reqType[req]) cb(new Error("Incorrect response type!"));
        else cb(null, d);
    });
};

Modchip.prototype.write = function (req,data) {
    var packet = Buffer(65);
    packet.fill(0);
    packet[1] = reqType[req];
    if (data) data.copy(packet,2);
    if (Modchip._debug) console.log("Trying to write", req, (0 && data) ? packet.toString('hex') : "");
    this._device.write(packet);
    if (Modchip._debug) console.log("Success.");
};

Modchip.prototype.read = function (cb) {
    if (Modchip._debug) console.log("Waiting for read…");
    this._device.read(function fn(e,d) {
        if (e) cb(e);
        else if (!d.length) dev.read(fn);
        else cb(e,d);
    });
};


// "static" methods

Modchip._debug = false;

Modchip.openModchip = function () {
    var devInfo = HID.devices().filter(function (d) {
        return (d.vendorId === vend && d.productId === prod);
    })[0];
    if (!devInfo) throw Error("Controller not connected!");
    else if (Modchip._debug) console.log("Found device:", devInfo);
    
    var dev = new HID.HID(devInfo.path);
    return new Modchip(dev);
};

Modchip.commands = Object.keys(reqType);


module.exports = Modchip;