var fmt_hex = require("./convert_hex"),
    util = require('util'),
    stream = require('stream');

function HexProgrammer(mcu, opts) {
    opts || (opts = {});
    stream.Transform.call(this, {objectMode:true});
    
    this._mcu = mcu;
    this._log = false;//!opts.silent;
    
    this._addr = null;
    this._writ = false;
    this._ALL_ = opts.writeAll || false;
    this.bytesWritten = 0;
}
util.inherits(HexProgrammer, stream.Transform);

HexProgrammer.prototype._transform = function (obj, enc, cb) {
    if ('startAddress' in obj) {
        if (this._writ) try {
            this._mcu.write('PROGRAM_COMPLETE');
        } catch (e) {
            return cb(e);
        }
        this._writ = false;
        this._addr = obj.startAddress;
    }
    if (Buffer.isBuffer(obj)) {
        var buff = obj.slice(0),
            range = fmt_hex._rangeForAddr(this._addr);
        
        var devRange = range && this._mcu.info.ranges.filter(function (dev) {
            return (dev.addr === range.start);
        })[0];
        if (devRange || this._ALL_) {
            if (devRange && !devRange._erased) try {
                this._mcu.write('ERASE_DEVICE', Buffer([devRange.idx]));
                devRange._erased = true;        // HACK: avoid erasing more than once
            } catch (e) {
                return cb(e);
            }
            
            if (this._log) console.log("WRITING",obj.length,"bytes @",'0x'+(this._addr/range.wdsz).toString(16));
            while (buff.length) {
                // NOTE: hex files always have 1-byte addressing, but not device
                var addr = this._addr / range.wdsz,
                    data = buff.slice(0,this._mcu._pgmBytes);
                try {
                    this._mcu.write('PROGRAM_DEVICE', HexProgrammer._makeProgPayload(this._mcu, addr, data));
                } catch (e) {
                    return cb(e);
                }
                this._writ = true;
                this._addr += data.length;
                this.bytesWritten += data.length;
                this.emit('activity');
                buff = buff.slice(data.length);
            }
        } else {
            if (this._log) console.log("SKIPPING",obj.length,"bytes @",'0x'+(this._addr/range.wdsz).toString(16));
            this._addr += obj.length;
        }
    }
    setImmediate(cb);
};

HexProgrammer.prototype._flush = function (cb) {
    if (this._writ) try {
        this._mcu.write('PROGRAM_COMPLETE');
    } catch (e) {
        return cb(e);
    }
    this._writ = false;
    this._addr = null;
    cb();
};


HexProgrammer._programStream = function (mcu, input, opts) {
    var hex = new fmt_hex.HexReader(),
        prg = new HexProgrammer(mcu, opts);
    input.pipe(hex).pipe(prg);
    hex.on('error', function (e) {
        e.badInput = true;
        prg.emit('error', e);
    });
    return prg;
};

HexProgrammer._programString = function (mcu, str, opts) {
    var input = new stream.PassThrough({objectMode:true}),
        prg = this._programStream(mcu, input, opts);
    input.end(str);
    return prg;
};

HexProgrammer._programBuffer = function (mcu, buff, opts) {
    var input = new stream.PassThrough({objectMode:true}),
        prg = new HexProgrammer(mcu, opts);
    if ('targetRange' in buff) {
        var range = fmt_hex._RANGES[buff.targetRange];
        if (!range) throw Error("Uknown target range '"+buff.targetRange+"'!");
        input.push({startAddress:range.start*range.wdsz});
    }
    input.end(buff);
    input.pipe(prg);
    return prg;
};

HexProgrammer.programDevice = function (mcu, data, opts) {
    opts || (opts = {});
    if (data instanceof stream.Readable) return this._programStream(mcu, data, opts);
    else if (typeof data === 'string') return this._programString(mcu, data, opts);
    else if (Buffer.isBuffer(data)) return this._programBuffer(mcu, data, opts);
    else throw Error("Uknown input type!");
};

HexProgrammer._makeProgPayload = function (mcu, addr, data) {
    if (data.length > mcu._pgmBytes) throw Error("Too much data to program!");
    
    var payload = Buffer(63),
        padding = mcu._pgmBytes - data.length;
    payload.fill(0);
    payload.writeUInt32LE(addr,0);      // Address
    payload[4] = data.length;           // Size
    payload.writeUInt16LE(0, 5);        // 2 PadBytes `(0x40-6) - 56` [why?? explains "right justification"…]
    data.copy(payload, 7+padding);
    return payload;    
}

module.exports = HexProgrammer;